/************************************************************************************
* This file contains necessary functions to access the SPI from Phy to Abel.
*
* Author(s):
*
* (c) Copyright 2004, Freescale, Inc.  All rights reserved.
*
* Freescale Confidential Proprietary
* Digianswer Confidential
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale.
*
* Last Inspected: 19-aug-2003
* Last Tested:
*
* Revision history:
*   date      Author    Comments
*   ------    ------    --------
*   200803     FFJ      Created
*   271003     TRL      Optimized phy_read_spi_1 and phy_write_spi_1
*   200904      JT      Optimized functions for Speed, Space and readability 
*   261004  JT/FFJ      Fixed BDM/SPI issue. (BDM is now allowed to cycle steal, while SPI is running) Optimized as well
*	131006      HS		The antenna switch control function modified 
*						the AssertCE and  DeAssertCE macro's commented out							
************************************************************************************/
#include <nucleus.h>
#include "hprfgw_rfEmbeddedTypes.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfMc1319xReg.h"
#include "hprfgw_rfPortConfig.h"
#include "hprfgw_rfMc1319xDrv.h"
#include "hprfgw_rfIrqControlLib.h"
#include "hardware.h"
#include "mcf5271_gpio.h"

/************************************************************************************
* External functions	 
************************************************************************************/
extern void EnableInterrupt(BOOL hi, UINT32 mask);
extern void DisableInterrupt(BOOL hi, UINT32 mask);

/************************************************************************************
* Local functions	 
************************************************************************************/
int IrqControlLib_IsMC1319xIrqAsserted(void);

/************************************************************************************
* Global Driver variables for getting MC1319x GPIO pin values 
************************************************************************************/
uint8_t* pMC1319xDrvPortGpio1  = (uint8_t*) &gMC1319xGPIO1Port; 
uint8_t* pMC1319xDrvPortGpio2  = (uint8_t*) &gMC1319xGPIO2Port;  
uint8_t  mMC1319xDrvMaskGpio1  = gMC1319xGPIO1Mask_c;
uint8_t  mMC1319xDrvMaskGpio2  = gMC1319xGPIO2Mask_c;
uint8_t  gStatusRegVal;

//#define	QSPI_DBG_IO
//#define	QSPI_DBG_IO2
//#define	QSPI_DBG_QUE	
typedef	struct	{
	UINT8	addr;
	UINT16	data;
	UINT8	direction;
}	TYPE_QSPI_DBG_QUE;
#define	QSPI_DBG_QUE_SIZE	250
typedef	struct	{
	TYPE_QSPI_DBG_QUE qs[QSPI_DBG_QUE_SIZE];
	UINT32	index;
	UINT32  cnt;
}	TYPE_QSPI_DBG_REC;
//TYPE_QSPI_DBG_REC QSPIR;

UINT8	QSPIaddr[QSPI_DBG_QUE_SIZE];
UINT16	QSPIdata[QSPI_DBG_QUE_SIZE];
UINT8	QSPIdirection[QSPI_DBG_QUE_SIZE];
UINT32	QSPIindex;
UINT32	QSPIcnt;

/************************************************************************************
* Local macros for controlling MC1319x GPIO pins
************************************************************************************/
/*
#define MC1319xDrv_AttEnable()      { gMC1319xAttnPort  |=  gMC1319xAttnMask_c;  }
#define MC1319xDrv_AttDisable()     { gMC1319xAttnPort  &= ~gMC1319xAttnMask_c;  }
#define MC1319xDrv_AssertReset()    { gMC1319xResetPort &= ~gMC1319xResetMask_c; }
#define MC1319xDrv_DeassertReset()  { gMC1319xResetPort |=  gMC1319xResetMask_c; }
*/
#define MC1319xDrv_AttEnable()      { MC1319xDrv_Att_Enable();		}
#define MC1319xDrv_AttDisable()     { MC1319xDrv_Att_Disable();	}
#define MC1319xDrv_AssertReset()    { MC1319xDrv_Assert_Reset();	}
#define MC1319xDrv_DeassertReset()  { MC1319xDrv_Deassert_Reset();	}

/************************************************************************************
* QSPI debug functions
************************************************************************************/
void MC1319xDrv_SaveQSPIData (UINT8 addr, UINT16 data, UINT8 direction)	{
	QSPIaddr[QSPIindex]		 = addr;
	QSPIdata[QSPIindex]		 = data;
	QSPIdirection[QSPIindex] = direction;
	if	(++QSPIindex >= QSPI_DBG_QUE_SIZE)	{	QSPIindex = 0;	}
	if	(++QSPIcnt > QSPI_DBG_QUE_SIZE)		{	QSPIcnt= QSPI_DBG_QUE_SIZE;	}

	
	/*

	if (++QSPIR.index >= QSPI_DBG_QUE_SIZE)	{	QSPIR.index = 0;					}
	if (++QSPIR.cnt   > QSPI_DBG_QUE_SIZE)	{	QSPIR.cnt   = QSPI_DBG_QUE_SIZE;	}

	QSPIR.qs[QSPIR.index].addr = addr;
	QSPIR.qs[QSPIR.index].data = data;
	QSPIR.qs[QSPIR.index].direction = direction;
*/
}

UINT16 MC1319xDrv_GetQSPIIndex (UINT16 *index)	{
	*index = QSPIindex;	 
	return (QSPIcnt);	
	
	
/*
	*index = QSPIR.index;
	return (QSPIR.cnt);
*/
}

void MC1319xDrv_GetQSPIData (UINT8 *addr, UINT16 *data, UINT8 *direction, UINT16 *index)	{
	*addr = QSPIaddr[*index];
	*data = QSPIdata[*index];
	*direction = QSPIdirection[*index];
	if	(*index)	{	*index -= 1;	}	else	{	*index = QSPI_DBG_QUE_SIZE-1;	}	

	
	/*
	*addr = QSPIR.qs[*index].addr;
	*data = QSPIR.qs[*index].data;	
	*direction = QSPIR.qs[*index].direction;	

	if (*index)	{	*index -= 1;	}	else	{	*index = QSPI_DBG_QUE_SIZE-1;	}	
	*/
}

/************************************************************************************
* Driver functions for controlling MC1319x GPIO pins
************************************************************************************/

void ackMC1319xIrq(void)	{
	
}

uint8_t isMC1319xActive(void)	{
	return (*pMC1319xDrvPortGpio1 & mMC1319xDrvMaskGpio1);
}

uint8_t isMC1319xDrvCrcOk(void)	{
	return (*pMC1319xDrvPortGpio2 & mMC1319xDrvMaskGpio2);
}

uint8_t isMC1319xDrvCcaBusy(void)	{
	return (*pMC1319xDrvPortGpio2 & mMC1319xDrvMaskGpio2);     
}

uint8_t isMC1319xIrqAsserted(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	return (!(eport->eport_pin_data_reg & MCF_EPORT_EPDDR_EPDD6));
}

void EnableMC1319xIrq(void)	{			// IRQ pin from MC1319x enabled
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	//	Enable interrupt 
	eport->eport_interrupt_ena_reg |= MCF_EPORT_EPFR_EPF6;
}
void DisableMC1319xIrq(void)	{		// IRQ pin from MC1319x disabled
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	int old_level;

	old_level = NU_Local_Control_Interrupts (NU_DISABLE_INTERRUPTS);
	eport->eport_interrupt_ena_reg &= ~MCF_EPORT_EPFR_EPF6;
	NU_Local_Control_Interrupts (old_level);
}

/*
void pushIrqStatus()	{
	asm(move.w SR,d7);
	asm(move.l d7,(sp));
    asm(suba #4,a7);
}

void pullIrqStatus()	{
	asm(adda #4,a7);
	asm(move.l (sp),d7);
	asm(move.w d7,SR);
}
*/

uint8_t returnEPIER(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	return (eport->eport_interrupt_ena_reg);	
	//return MCF_EPORT_EPIER;
}

void restoreEPIER(uint8_t var)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	if (var & MCF_EPORT_EPIER_EPIE6)	{
		EnableMC1319xIrq();
	}	else	{
		DisableMC1319xIrq();	}
	//MCF_EPORT_EPIER |= (var&MCF_EPORT_EPIER_EPIE6);
}

void InitMC1319xIrq	(void)	{
	EnableMC1319xIrq();
	//MCF_EPORT_EPIER |= MCF_EPORT_EPIER_EPIE1;
}

//========  End of new calls =================================


void MC1319xDrv_TxHighPowerEnable(void)	{
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  |= gMC1319xPAEnMask_c;
#else
	gpio->podr_datah  |= gMC1319xPAEnMask_c;
#endif
}

void MC1319xDrv_TxHighPowerDisable(void)	{
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  &= ~gMC1319xPAEnMask_c;
#else
	gpio->podr_datah  &= ~gMC1319xPAEnMask_c;
#endif
}

void MC1319xDrv_RxTxEnable(void)            { 
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  |= gMC1319xRxTxMask_c;
#else
	gpio->podr_datah  |= gMC1319xRxTxMask_c;
#endif
}
void MC1319xDrv_RxTxDisable(void)           { 
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  &= ~gMC1319xRxTxMask_c;
#else
	gpio->podr_datah  &= ~gMC1319xRxTxMask_c;
#endif
}
void MC1319xDrv_RxAntennaSwitchEnable(void) { 
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
	MC1319xDrv_TxHighPowerDisable();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  |= gMC1319xAntSwTxEnMask_c;
	gpio->podr_datal  &= ~gMC1319xAntSwRxEnMask_c;
#else
	gpio->podr_datah  |= gMC1319xAntSwTxEnMask_c;
	gpio->podr_datah  &= ~gMC1319xAntSwRxEnMask_c;
#endif
}
void MC1319xDrv_TxAntennaSwitchEnable(void) { 
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
	MC1319xDrv_TxHighPowerEnable();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  |= gMC1319xAntSwRxEnMask_c;
	gpio->podr_datal  &= ~gMC1319xAntSwTxEnMask_c;
#else
	gpio->podr_datah  |= gMC1319xAntSwRxEnMask_c;
	gpio->podr_datah  &= ~gMC1319xAntSwTxEnMask_c;
#endif
}

void MC1319xDrv_RxLnaDisable(void)          {/* Customer specific code. Very fast execution required */}
void MC1319xDrv_TxPaDisable(void)           {/* Customer specific code. Very fast execution required */}

void MC1319xDrv_Assert_Reset(void)	{
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  &= ~gMC1319xResetMask_c;
#else
	gpio->podr_datah  &= ~gMC1319xResetMask_c;
#endif
}

void MC1319xDrv_Deassert_Reset(void)	{
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal |= gMC1319xResetMask_c;
#else
	gpio->podr_datah |= gMC1319xResetMask_c;
#endif
}

void MC1319xDrv_Att_Enable(void)	{	
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  |= gMC1319xAttnMask_c;
#else
	gpio->podr_datah  |= gMC1319xAttnMask_c;
#endif
}

void MC1319xDrv_Att_Disable (void) { 
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	gpio->podr_datal  &= ~gMC1319xAttnMask_c;
#else
	gpio->podr_datah  &= ~gMC1319xAttnMask_c;
#endif
}


#ifdef	ENABLE_OLD_CALLS

UINT8 MC1319xDrv_IsActive(void)	{	
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	return((UINT32) (gpio->ppdsdr_datal&gMC1319xGPIO1Mask_c));
#else
	return((UINT32) (gpio->ppdsdr_cs&gMC1319xGPIO1Mask_c));
#endif
}

UINT8 MC1319xDrv_IsCrcOk(void)            {
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	return((UINT32) (gpio->ppdsdr_datal&gMC1319xGPIO2Mask_c));
#else
	return((UINT32) (gpio->ppdsdr_cs&gMC1319xGPIO2Mask_c));
#endif
}    

UINT8 MC1319xDrv_IsCcaBusy(void)            {
	GPIO_5271_Regs *gpio = mcf5271_gpio_ptr();
#ifdef	TPS6_CLIENT_TEST
	return((UINT32) (gpio->ppdsdr_datal&gMC1319xGPIO2Mask_c));	
#else
	return((UINT32) (gpio->ppdsdr_cs&gMC1319xGPIO2Mask_c));
#endif
}   

#endif

/************************************************************************************
*  SPI functions											                        *
************************************************************************************/

/************************************************************************************
*  SPI initialization for MC1319x driver interface                                  *
************************************************************************************/
void MC1319xDrv_Init(void)	{
  MC1319xDrv_AssertReset();    // Keep MC1319x in reset for now
}

/*******************************************************************
* Make a pulse on hardware reset PIN at ABEL                       *
*******************************************************************/
void MC1319xDrv_Reset(void)
{
  int i;
  uint16_t dummyClear;
	
  MC1319xDrv_AssertReset();			
  HwDelayMsec(10);
  MC1319xDrv_DeassertReset();
  HwDelayMsec(10);
  MC1319xDrv_WriteSpi(0x00, 0x0000);/*soft reset*/
  HwDelayMsec(30);  
  
  while(!IrqControlLib_IsMC1319xIrqAsserted());
  IrqControlLib_AckMC1319xIrq();
  MC1319xDrv_ReadSpiAsync(ABEL_reg24, &dummyClear);
}


/*******************************************************************************
* Pulse ATTN pin of the MC1319x in order to bring it out of power save mode    *
*******************************************************************************/
void MC1319xDrv_WakeUp(void)
{
  uint8_t i=10; // ~10us ATTN pulse @ 62.5KHz

  MC1319xDrv_AttEnable();

  while(i) { i--; }

  MC1319xDrv_AttDisable();

}

/************************************************************************************
* Local macros for SPI functions
* will not be used by CF qspi
************************************************************************************/

/************************************************************************************
* Local macros
* will not be used by cf using inlining functions in .h file.
************************************************************************************/

/*************************************************************************
*                MC13192x IRQ interrupt control functions		         *
*************************************************************************/

void IrqControlLib_DisableMC1319xIrq(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	int old_level;

	old_level = NU_Local_Control_Interrupts(NU_DISABLE_INTERRUPTS);	
	eport->eport_interrupt_ena_reg &= ~MCF_EPORT_EPFR_EPF6;
	NU_Local_Control_Interrupts(old_level);
}

int IrqControlLib_IsMC1319xIrqAsserted(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	return (!(eport->eport_pin_data_reg & MCF_EPORT_EPFR_EPF6));
}

#ifdef	ENABLE_OLD_CALLS

void IrqControlLib_EnableMC1319xIrq(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();

	//	Clear pending int flag, enable interrupt 
	eport->eport_interrupt_ena_reg |= MCF_EPORT_EPFR_EPF6;
}	

void IrqControlLib_DisableMC1319xIrq(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	int old_level;

	old_level = NU_Local_Control_Interrupts(NU_DISABLE_INTERRUPTS);	
	eport->eport_interrupt_ena_reg &= ~MCF_EPORT_EPFR_EPF6;
	NU_Local_Control_Interrupts(old_level);
}

void IrqControlLib_InitMC1319xIrq(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	eport->eport_interrupt_ena_reg |= MCF_EPORT_EPFR_EPF6;
}

int IrqControlLib_IsMC1319xIrqAsserted(void)	{
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	return (!(eport->eport_pin_data_reg & MCF_EPORT_EPFR_EPF6));
}

#endif

UINT8 IrqControlLib_BackupAndDisableIntControl (void)	{
	UINT8 var;
	//INTCx_5271_Regs *intc0 = mcf5271_intc0_ptr();
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	int old_level;

	//var = intc0->int_mask_low;   
	var = eport->eport_interrupt_ena_reg;
	old_level = NU_Local_Control_Interrupts (NU_DISABLE_INTERRUPTS);
	//DisableInterrupt(FALSE,  MCF_IMRL_EXT_INT6);
	eport->eport_interrupt_ena_reg &= ~MCF_EPORT_EPFR_EPF6;
	NU_Local_Control_Interrupts(old_level);

	//return ((var&MCF_IMRL_EXT_INT6)?FALSE:TRUE);
	return ((var&MCF_EPORT_EPFR_EPF6)?TRUE:FALSE);
}

void	IrqControlLib_RestoreIntControl (UINT8 var)	{
	//INTCx_5271_Regs *intc0 = mcf5271_intc0_ptr();
	EPORT_5271_Regs* eport = mcf5271_eport_ptr();
	int old_level;

	if (var)	{
		//EnableInterrupt(FALSE,  MCF_IMRL_EXT_INT6);
		eport->eport_interrupt_ena_reg |= MCF_EPORT_EPFR_EPF6;
	}	else	{
		old_level = NU_Local_Control_Interrupts(NU_DISABLE_INTERRUPTS);
		eport->eport_interrupt_ena_reg &= ~MCF_EPORT_EPFR_EPF6;
		//DisableInterrupt(FALSE,  MCF_IMRL_EXT_INT6);
		//intc0->int_mask_low |= (long)(MCF_INTC0_IMRL_INT_MASK6);
		NU_Local_Control_Interrupts(old_level);
	}	
}


#ifdef	ENABLE_OLD_CALLS
UINT8	IrqControlLib_ProtectFromMC1319xIrq(void)	{
	return (IrqControlLib_BackupAndDisableIntControl());
}  

void IrqControlLib_UnprotectFromMC1319xIrq(UINT8 var)	{
	IrqControlLib_RestoreIntControl(var);
}	
#endif

/*************************************************************************
*                       SPI Read functions                               *
*************************************************************************/
/*****************************************************************
* This function reads 1 16bit Abel Register at address "addr"    *
* This read may be called from anywhere. (Async and sync context)*
*****************************************************************/
void MC1319xDrv_ReadSpiIntCore (uint8_t addr, uint16_t *pb)	{

	QSPI_5271_Regs *qspi = get_qspi_ptr();

	qspi->qspi_addr_reg  = 0;
	qspi->qspi_data_reg  = (addr | 0x80);
	qspi->qspi_delay_reg = 0x8000;
	while (!(qspi->qspi_int_reg & MCF_QSPI_QIR_SPIF));
	qspi->qspi_int_reg	 = 0xD00F;
	qspi->qspi_addr_reg  = 0x10;
	qspi->qspi_data_reg;
	*pb = qspi->qspi_data_reg;

#ifdef	QSPI_DBG_QUE
	MC1319xDrv_SaveQSPIData (addr, *pb, 0);	
#endif
}

void MC1319xDrv_ReadSpiIntCoreLE (uint8_t addr, uint16_t *pb)	{
	QSPI_5271_Regs *qspi = get_qspi_ptr();
	uint16_t temporary;

	qspi->qspi_addr_reg  = 0;
	qspi->qspi_data_reg  = (addr | 0x80);
	qspi->qspi_delay_reg = 0x8000;
	while (!(qspi->qspi_int_reg & MCF_QSPI_QIR_SPIF));
	qspi->qspi_int_reg	 = 0xD00F;
	qspi->qspi_addr_reg  = 0x10;
	qspi->qspi_data_reg;
	temporary = qspi->qspi_data_reg;
	*pb = temporary << 8;								
	*pb = *pb | (temporary >> 8);		
}

/*****************************************************************
* This function reads 1 16bit Abel Register at address "addr"    *
* This read may be called from anywhere. (Async and sync context)*
*****************************************************************/
void MC1319xDrv_ReadSpi(uint8_t addr, uint16_t *pb)	{
	int old_level;

	old_level = NU_Local_Control_Interrupts (NU_DISABLE_INTERRUPTS);
	MC1319xDrv_ReadSpiIntCore(addr, pb);
	NU_Local_Control_Interrupts (old_level);
}

/****************************************************************
* This function reads 1 16bit Abel Register at address "addr"   *
* This read is only to be used from Abel interrupt context      *
****************************************************************/
void MC1319xDrv_ReadSpiInt(uint8_t addr, uint16_t *pb)	{
  int	old_level;

	old_level = NU_Local_Control_Interrupts(NU_DISABLE_INTERRUPTS);	
	MC1319xDrv_ReadSpiIntCore(addr, pb); 
	NU_Local_Control_Interrupts(old_level);
}

/****************************************************************
* This function reads 1 16bit Abel Register at address "addr"   *
* This read is only to be used from Abel interrupt context      *
* Note: Forced Little Endianess!                                *
****************************************************************/
void MC1319xDrv_ReadSpiIntLE(uint8_t addr, uint16_t *pb)	{
	int	old_level;

	old_level = NU_Local_Control_Interrupts(NU_DISABLE_INTERRUPTS);
	MC1319xDrv_ReadSpiIntCoreLE(addr, pb); 
	NU_Local_Control_Interrupts(old_level);
}


/*************************************************************************
*                      SPI Write functions                               *
*************************************************************************/

/********************************************************************
* Write one 16bit data-location into Abel at address "addr"         *
* without monitoring SPSCR. Data is transferred to this routine     *
* over the stack.                                                   *
* Unprotected - only to be called with disabled interrupts          *
* Macro used in below functions                                     *
********************************************************************/
void MC1319xDrv_WriteSpiIntCore (uint8_t addr, uint16_t content)	{
	QSPI_5271_Regs *qspi = get_qspi_ptr();

	if (addr == 6)	{	if (content == 0)	{	QSPIcnt++;	}	}	

	qspi->qspi_addr_reg  = 0;
	qspi->qspi_data_reg  = addr;
	qspi->qspi_data_reg  = content;
	qspi->qspi_delay_reg = 0x8000;
	while (!(qspi->qspi_int_reg & MCF_QSPI_QIR_SPIF));
	qspi->qspi_int_reg   = (0xD00F);

#ifdef	QSPI_DBG_QUE	
	MC1319xDrv_SaveQSPIData (addr, content, 1);
#endif
}

/********************************************************************
* Write one 16bit data-location into Abel at address "addr"         *
* without monitoring SPSCR.                                         *
* This write is only to be used from Abel interrupt context         *
* Protected!                                                        *
********************************************************************/
void MC1319xDrv_WriteSpiInt(uint8_t addr, uint16_t content)	{
	int	old_level;

	old_level = NU_Local_Control_Interrupts(NU_DISABLE_INTERRUPTS);
	MC1319xDrv_WriteSpiIntCore (addr, content);  
	NU_Local_Control_Interrupts(old_level);
}

/********************************************************************
* Write one 16bit data-location into Abel at address "addr"         *
* without monitoring SPSCR.                                         *
* This write is only to be used from Abel interrupt context         *
* Not Protected!                                                    *
********************************************************************/
void MC1319xDrv_WriteSpiIntFast(uint8_t addr, uint16_t content)	{
	int   old_level;

	//old_level = NU_Local_Control_Interrupts (NU_DISABLE_INTERRUPTS);
	MC1319xDrv_WriteSpiIntCore(addr, content); 
	//NU_Local_Control_Interrupts (old_level);
}

/********************************************************************
* Write one 16bit data-location into Abel at address "addr"         *
* without monitoring SPSCR. Data is transferred to this routine     *
* over the stack.													*
*																	*	
* Note:																*
* function called from interrupt, this function is not-preemptive	*
* or reentrant														*	
********************************************************************/
void MC1319xDrv_WriteSpi(uint8_t addr, uint16_t content)	{
	int   old_level;

	old_level = NU_Local_Control_Interrupts (NU_DISABLE_INTERRUPTS);
	MC1319xDrv_WriteSpiIntFast(addr, content); 
	NU_Local_Control_Interrupts (old_level);
}

